// ========================================
// Navigation
// ========================================
const navbar = document.getElementById('navbar');
const hamburger = document.getElementById('hamburger');
const navMenu = document.getElementById('nav-menu');
const navLinks = document.querySelectorAll('.nav-link');

// Navbar scroll effect
window.addEventListener('scroll', () => {
    if (window.scrollY > 100) {
        navbar.classList.add('scrolled');
    } else {
        navbar.classList.remove('scrolled');
    }
});

// Mobile menu toggle
if (hamburger) {
    hamburger.addEventListener('click', () => {
        hamburger.classList.toggle('active');
        navMenu.classList.toggle('active');
    });
}

// Close mobile menu when clicking on a link
navLinks.forEach(link => {
    link.addEventListener('click', () => {
        hamburger?.classList.remove('active');
        navMenu?.classList.remove('active');
    });
});

// ========================================
// Back to Top Button
// ========================================
const backToTopBtn = document.getElementById('back-to-top');

if (backToTopBtn) {
    window.addEventListener('scroll', () => {
        if (window.scrollY > 500) {
            backToTopBtn.classList.add('visible');
        } else {
            backToTopBtn.classList.remove('visible');
        }
    });

    backToTopBtn.addEventListener('click', () => {
        window.scrollTo({
            top: 0,
            behavior: 'smooth'
        });
    });
}

// ========================================
// Scroll Animations
// ========================================
const observerOptions = {
    threshold: 0.2,
    rootMargin: '0px 0px -100px 0px'
};

const observer = new IntersectionObserver((entries) => {
    entries.forEach(entry => {
        if (entry.isIntersecting) {
            entry.target.classList.add('animated');
        }
    });
}, observerOptions);

// Observe all elements with animate-on-scroll class
document.querySelectorAll('.animate-on-scroll').forEach(el => {
    observer.observe(el);
});

// ========================================
// Gallery Preview - Home Page
// ========================================
const galleryGrid = document.getElementById('gallery-grid');

if (galleryGrid) {
    // Najbolje slike (najviša rezolucija) iz image_manifest.json
    const galleryImages = [
        {
            src: 'downloaded_images/09_veselja_2018/01_2.jpg',
            caption: 'Veselje 2018'
        },
        {
            src: 'downloaded_images/09_veselja_2017/01_2.jpg',
            caption: 'Veselje 2017'
        },
        {
            src: 'downloaded_images/09_veselja_2016/01_6.jpg',
            caption: 'Veselje 2016'
        },
        {
            src: 'downloaded_images/05_dekoracija/01.jpg',
            caption: 'Dekoracija'
        },
        {
            src: 'downloaded_images/10_ostalo_beli-satori/glavna.jpg',
            caption: 'Beli Šatori'
        },
        {
            src: 'downloaded_images/10_ostalo_meni-hrana/1.jpg',
            caption: 'Hrana'
        },
        {
            src: 'downloaded_images/10_ostalo_meni-hrana/2.jpg',
            caption: 'Hrana'
        },
        {
            src: 'downloaded_images/10_ostalo_meni-torte/1.jpg',
            caption: 'Torte'
        },
        {
            src: 'downloaded_images/10_ostalo_pagode/01.jpg',
            caption: 'Pagode'
        },
        {
            src: 'downloaded_images/08_voce/0.jpg',
            caption: 'Voće'
        },
        {
            src: 'downloaded_images/08_voce/1.jpg',
            caption: 'Voće'
        },
        {
            src: 'downloaded_images/09_veselja_2015/01_3.jpg',
            caption: 'Veselje 2015'
        },
        {
            src: 'downloaded_images/05_dekoracija/02.jpg',
            caption: 'Dekoracija'
        },
        {
            src: 'downloaded_images/10_ostalo_meni-hrana/3.jpg',
            caption: 'Hrana'
        },
        {
            src: 'downloaded_images/10_ostalo_meni-torte/2.jpg',
            caption: 'Torte'
        },
        {
            src: 'downloaded_images/09_veselja_2014/01_1.jpg',
            caption: 'Veselje 2014'
        }
    ];

    // Load images with error handling
    galleryImages.forEach((img, index) => {
        // Test if image exists before adding to gallery
        const testImg = new Image();
        testImg.onload = function() {
            const item = document.createElement('div');
            item.className = 'gallery-item animate-on-scroll';
            item.style.animationDelay = `${index * 0.1}s`;
            item.innerHTML = `
                <img src="${img.src}" alt="${img.caption}" loading="lazy">
                <div class="gallery-overlay">
                    <span class="gallery-caption">${img.caption}</span>
                </div>
            `;
            galleryGrid.appendChild(item);
            if (typeof observer !== 'undefined') {
                observer.observe(item);
            }
        };
        testImg.onerror = function() {
            console.log('Failed to load:', img.src);
        };
        testImg.src = img.src;
    });
    
}

// ========================================
// Lightbox for Gallery
// ========================================
let lightbox = null;

function createLightbox() {
    if (lightbox) return lightbox;
    
    lightbox = document.createElement('div');
    lightbox.className = 'lightbox';
    lightbox.innerHTML = `
        <div class="lightbox-overlay"></div>
        <div class="lightbox-content">
            <button class="lightbox-close" aria-label="Close">&times;</button>
            <button class="lightbox-prev" aria-label="Previous">&lsaquo;</button>
            <button class="lightbox-next" aria-label="Next">&rsaquo;</button>
            <img src="" alt="">
            <div class="lightbox-caption"></div>
        </div>
    `;
    
    document.body.appendChild(lightbox);
    
    // Add lightbox styles
    const style = document.createElement('style');
    style.textContent = `
        .lightbox {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            z-index: 9999;
            align-items: center;
            justify-content: center;
        }
        
        .lightbox.active {
            display: flex;
        }
        
        .lightbox-overlay {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.95);
            cursor: pointer;
        }
        
        .lightbox-content {
            position: relative;
            max-width: 90%;
            max-height: 90%;
            z-index: 1;
        }
        
        .lightbox-content img {
            max-width: 100%;
            max-height: 90vh;
            object-fit: contain;
            border-radius: 8px;
        }
        
        .lightbox-close {
            position: absolute;
            top: -50px;
            right: 0;
            background: none;
            border: none;
            color: white;
            font-size: 3rem;
            cursor: pointer;
            line-height: 1;
            padding: 10px;
            transition: transform 0.2s;
        }
        
        .lightbox-close:hover {
            transform: scale(1.2);
        }
        
        .lightbox-prev,
        .lightbox-next {
            position: absolute;
            top: 50%;
            transform: translateY(-50%);
            background: rgba(255, 255, 255, 0.1);
            border: 2px solid white;
            color: white;
            font-size: 3rem;
            cursor: pointer;
            padding: 10px 20px;
            border-radius: 8px;
            transition: all 0.3s;
        }
        
        .lightbox-prev {
            left: -80px;
        }
        
        .lightbox-next {
            right: -80px;
        }
        
        .lightbox-prev:hover,
        .lightbox-next:hover {
            background: rgba(255, 255, 255, 0.2);
            transform: translateY(-50%) scale(1.1);
        }
        
        .lightbox-caption {
            position: absolute;
            bottom: -50px;
            left: 0;
            right: 0;
            color: white;
            text-align: center;
            font-size: 1.25rem;
            font-weight: 500;
        }
        
        @media (max-width: 768px) {
            .lightbox-prev,
            .lightbox-next {
                font-size: 2rem;
                padding: 5px 15px;
            }
            
            .lightbox-prev {
                left: 10px;
            }
            
            .lightbox-next {
                right: 10px;
            }
            
            .lightbox-close {
                top: 10px;
                right: 10px;
            }
        }
    `;
    document.head.appendChild(style);
    
    return lightbox;
}

function openLightbox(imageSrc, caption, images = [], currentIndex = 0) {
    const lb = createLightbox();
    const img = lb.querySelector('img');
    const captionEl = lb.querySelector('.lightbox-caption');
    const closeBtn = lb.querySelector('.lightbox-close');
    const prevBtn = lb.querySelector('.lightbox-prev');
    const nextBtn = lb.querySelector('.lightbox-next');
    const overlay = lb.querySelector('.lightbox-overlay');
    
    img.src = imageSrc;
    captionEl.textContent = caption;
    lb.classList.add('active');
    document.body.style.overflow = 'hidden';
    
    let currentIdx = currentIndex;
    
    function updateImage() {
        if (images.length > 0) {
            img.src = images[currentIdx].src;
            captionEl.textContent = images[currentIdx].caption;
        }
    }
    
    function close() {
        lb.classList.remove('active');
        document.body.style.overflow = '';
    }
    
    closeBtn.onclick = close;
    overlay.onclick = close;
    
    if (images.length > 1) {
        prevBtn.style.display = 'block';
        nextBtn.style.display = 'block';
        
        prevBtn.onclick = (e) => {
            e.stopPropagation();
            currentIdx = (currentIdx - 1 + images.length) % images.length;
            updateImage();
        };
        
        nextBtn.onclick = (e) => {
            e.stopPropagation();
            currentIdx = (currentIdx + 1) % images.length;
            updateImage();
        };
        
        // Keyboard navigation
        document.addEventListener('keydown', function navKeys(e) {
            if (!lb.classList.contains('active')) {
                document.removeEventListener('keydown', navKeys);
                return;
            }
            
            if (e.key === 'ArrowLeft') prevBtn.click();
            if (e.key === 'ArrowRight') nextBtn.click();
            if (e.key === 'Escape') close();
        });
    } else {
        prevBtn.style.display = 'none';
        nextBtn.style.display = 'none';
        
        document.addEventListener('keydown', function escKey(e) {
            if (e.key === 'Escape') {
                close();
                document.removeEventListener('keydown', escKey);
            }
        });
    }
}

// Attach lightbox to gallery items
document.addEventListener('click', (e) => {
    const galleryItem = e.target.closest('.gallery-item');
    if (galleryItem) {
        const img = galleryItem.querySelector('img');
        const caption = galleryItem.querySelector('.gallery-caption')?.textContent || '';
        
        // Get all gallery items for navigation
        const allItems = Array.from(document.querySelectorAll('.gallery-item'));
        const images = allItems.map(item => ({
            src: item.querySelector('img').src,
            caption: item.querySelector('.gallery-caption')?.textContent || ''
        }));
        
        const currentIndex = allItems.indexOf(galleryItem);
        
        openLightbox(img.src, caption, images, currentIndex);
    }
});

// ========================================
// Form Validation (for contact page)
// ========================================
const contactForm = document.getElementById('contact-form');

if (contactForm) {
    contactForm.addEventListener('submit', async (e) => {
        e.preventDefault();
        
        const formData = new FormData(contactForm);
        const data = Object.fromEntries(formData);
        
        // Basic validation
        if (!data.name || !data.email || !data.message) {
            showNotification('Molimo popunite sva obavezna polja', 'error');
            return;
        }
        
        // Email validation
        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        if (!emailRegex.test(data.email)) {
            showNotification('Molimo unesite validnu email adresu', 'error');
            return;
        }
        
        // Show success message (in real app, send to server)
        showNotification('Hvala! Vaša poruka je uspešno poslata. Javićemo vam se uskoro.', 'success');
        contactForm.reset();
    });
}

function showNotification(message, type = 'success') {
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.textContent = message;
    
    const style = document.createElement('style');
    if (!document.getElementById('notification-styles')) {
        style.id = 'notification-styles';
        style.textContent = `
            .notification {
                position: fixed;
                top: 100px;
                right: 20px;
                padding: 1rem 1.5rem;
                border-radius: 8px;
                box-shadow: 0 4px 12px rgba(0,0,0,0.2);
                z-index: 10000;
                animation: slideInRight 0.3s ease;
                max-width: 400px;
            }
            
            .notification-success {
                background: #10b981;
                color: white;
            }
            
            .notification-error {
                background: #ef4444;
                color: white;
            }
            
            @keyframes slideInRight {
                from {
                    transform: translateX(400px);
                    opacity: 0;
                }
                to {
                    transform: translateX(0);
                    opacity: 1;
                }
            }
            
            @keyframes slideOutRight {
                from {
                    transform: translateX(0);
                    opacity: 1;
                }
                to {
                    transform: translateX(400px);
                    opacity: 0;
                }
            }
        `;
        document.head.appendChild(style);
    }
    
    document.body.appendChild(notification);
    
    setTimeout(() => {
        notification.style.animation = 'slideOutRight 0.3s ease';
        setTimeout(() => notification.remove(), 300);
    }, 4000);
}

// ========================================
// Lazy Loading Images
// ========================================
if ('IntersectionObserver' in window) {
    const imageObserver = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                const img = entry.target;
                if (img.dataset.src) {
                    img.src = img.dataset.src;
                    img.removeAttribute('data-src');
                }
                imageObserver.unobserve(img);
            }
        });
    });
    
    document.querySelectorAll('img[data-src]').forEach(img => {
        imageObserver.observe(img);
    });
}

// ========================================
// Smooth Scroll for Anchor Links
// ========================================
document.querySelectorAll('a[href^="#"]').forEach(anchor => {
    anchor.addEventListener('click', function(e) {
        const href = this.getAttribute('href');
        if (href === '#') return;
        
        e.preventDefault();
        const target = document.querySelector(href);
        
        if (target) {
            const offsetTop = target.offsetTop - 80;
            window.scrollTo({
                top: offsetTop,
                behavior: 'smooth'
            });
        }
    });
});

// ========================================
// Page Load Animations
// ========================================
window.addEventListener('load', () => {
    document.body.classList.add('loaded');
});

// ========================================
// Active Page Indicator
// ========================================
const currentPage = window.location.pathname.split('/').pop() || 'index.html';
document.querySelectorAll('.nav-link').forEach(link => {
    const linkPage = link.getAttribute('href');
    if (linkPage === currentPage || (currentPage === '' && linkPage === 'index.html')) {
        link.classList.add('active');
    } else {
        link.classList.remove('active');
    }
});

console.log('Šatori Foto Goran - Website loaded successfully!');

