// Gallery Loader - Dynamically loads images from downloaded_images directory

// Function to dynamically scan and load all images from a folder
function loadGalleryFromFolder(galleryId, folderPath, categoryName, maxImages = 100) {
    const gallery = document.getElementById(galleryId);
    if (!gallery) return;
    
    const commonImageNames = [];
    
    // Generate common naming patterns
    for (let i = 0; i <= maxImages; i++) {
        const num = i.toString().padStart(2, '0');
        commonImageNames.push(`${num}.jpg`);
        // Also try variants with _1, _2, etc
        for (let j = 1; j <= 20; j++) {
            commonImageNames.push(`${num}_${j}.jpg`);
        }
    }
    
    let loadedCount = 0;
    
    // Try to load images
    commonImageNames.forEach((imgName, index) => {
        const imgPath = `${folderPath}/${imgName}`;
        
        // Create a test image to check if it exists
        const testImg = new Image();
        testImg.onload = function() {
            const item = document.createElement('div');
            item.className = 'gallery-item animate-on-scroll';
            item.style.animationDelay = `${loadedCount * 0.05}s`;
            
            item.innerHTML = `
                <img src="${imgPath}" alt="${categoryName}" loading="lazy">
                <div class="gallery-overlay">
                    <span class="gallery-caption">${categoryName}</span>
                </div>
            `;
            
            gallery.appendChild(item);
            loadedCount++;
            
            // Observe for scroll animations
            if (typeof observer !== 'undefined') {
                observer.observe(item);
            }
        };
        
        testImg.src = imgPath;
    });
}

// Simplified version for specific image lists
function loadGallery(galleryId, imageList, categoryName) {
    const gallery = document.getElementById(galleryId);
    if (!gallery) return;
    
    imageList.forEach((imgPath, index) => {
        const item = document.createElement('div');
        item.className = 'gallery-item animate-on-scroll';
        item.style.animationDelay = `${index * 0.05}s`;
        
        item.innerHTML = `
            <img src="${imgPath}" alt="${categoryName}" loading="lazy" onerror="this.parentElement.remove()">
            <div class="gallery-overlay">
                <span class="gallery-caption">${categoryName}</span>
            </div>
        `;
        
        gallery.appendChild(item);
        
        // Observe for scroll animations
        if (typeof observer !== 'undefined') {
            observer.observe(item);
        }
    });
}

// Auto-load galleries based on page
const currentPath = window.location.pathname;

// Beli Satori Page
if (currentPath.includes('beli-satori') || document.getElementById('satori-gallery')) {
    loadGalleryFromFolder('satori-gallery', 'downloaded_images/10_ostalo_beli-satori', 'Beli Šatori', 60);
}

// Pagode Page
if (currentPath.includes('pagode') || document.getElementById('pagode-gallery')) {
    loadGalleryFromFolder('pagode-gallery', 'downloaded_images/10_ostalo_pagode', 'Pagode', 50);
}

// Main Gallery Page - Load all categories
if (currentPath.includes('galerija') || document.getElementById('main-gallery')) {
    const mainGallery = document.getElementById('main-gallery');
    
    if (mainGallery) {
        // Load all wedding galleries
        const weddingYears = [2018, 2017, 2016, 2015, 2014, 2013];
        weddingYears.forEach(year => {
            loadGalleryFromFolder('main-gallery', `downloaded_images/09_veselja_${year}`, `Veselja ${year}`, 20);
        });
        
        // Load other categories
        loadGalleryFromFolder('main-gallery', 'downloaded_images/05_dekoracija', 'Dekoracija', 25);
        loadGalleryFromFolder('main-gallery', 'downloaded_images/10_ostalo_beli-satori', 'Beli Šatori', 30);
        loadGalleryFromFolder('main-gallery', 'downloaded_images/10_ostalo_pagode', 'Pagode', 30);
        loadGalleryFromFolder('main-gallery', 'downloaded_images/10_ostalo_meni-hrana', 'Hrana', 20);
        loadGalleryFromFolder('main-gallery', 'downloaded_images/10_ostalo_meni-torte', 'Torte', 15);
        loadGalleryFromFolder('main-gallery', 'downloaded_images/08_voce', 'Voće', 20);
    }
}

// Add CSS for masonry gallery
if (document.getElementById('satori-gallery') || document.getElementById('pagode-gallery')) {
    const style = document.createElement('style');
    style.textContent = `
        .masonry-gallery {
            column-count: 3;
            column-gap: 1.5rem;
        }
        
        .masonry-gallery .gallery-item {
            break-inside: avoid;
            margin-bottom: 1.5rem;
            height: auto;
        }
        
        @media (max-width: 1024px) {
            .masonry-gallery {
                column-count: 2;
            }
        }
        
            @media (max-width: 768px) {
                .masonry-gallery {
                    column-count: 1;
                }
                
                .page-header {
                    padding: 150px 0 80px;
                    min-height: 350px;
                }
            }
        
        .page-header {
            position: relative;
            padding: 180px 0 100px;
            min-height: 500px;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
        }
        
        .page-header-bg {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-size: cover;
            background-position: center;
            z-index: -2;
        }
        
        .page-header-overlay {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: linear-gradient(135deg, rgba(44, 62, 80, 0.85), rgba(201, 169, 97, 0.75));
            z-index: -1;
        }
        
        .page-header-content {
            text-align: center;
        }
        
        .page-title {
            font-size: clamp(2.5rem, 5vw, 4rem);
            margin-bottom: 1rem;
        }
        
        .page-subtitle {
            font-size: clamp(1.125rem, 2vw, 1.5rem);
            opacity: 0.95;
        }
        
        .content-wrapper {
            max-width: 1200px;
            margin: 0 auto;
        }
        
        .intro-content {
            margin-top: 3rem;
        }
        
        .features-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 2rem;
            margin-top: 3rem;
        }
        
        .feature-box {
            background: white;
            padding: 2rem;
            border-radius: 16px;
            box-shadow: 0 4px 8px rgba(0,0,0,0.1);
            text-align: center;
            transition: all 0.3s ease;
        }
        
        .feature-box:hover {
            transform: translateY(-10px);
            box-shadow: 0 12px 24px rgba(0,0,0,0.15);
        }
        
        .feature-icon-large {
            font-size: 4rem;
            margin-bottom: 1rem;
        }
        
        .feature-box h3 {
            color: var(--color-secondary);
            margin-bottom: 0.5rem;
        }
        
        .feature-box p {
            color: var(--color-gray);
            margin: 0;
        }
        
        .specs-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
            gap: 2rem;
        }
        
        .spec-card {
            background: var(--color-gray-light);
            padding: 2rem;
            border-radius: 12px;
            border-left: 4px solid var(--color-primary);
        }
        
        .spec-card h3 {
            color: var(--color-primary);
            margin-bottom: 1rem;
        }
        
        .spec-card p {
            color: var(--color-gray-dark);
            margin: 0;
        }
    `;
    document.head.appendChild(style);
}

